# coding: utf-8
#
# Copyright (c) 2024 Nebil A. Katcho, Juan Rodríguez Carvajal
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

__authors__ = ["Nebil A. Katcho","Juan Rodriguez-Carvajal"]
__license__ = "MIT"
__date__    = "19/05/2025"

import os
import screeninfo
from PySide6 import QtCore,QtGui,QtWidgets
from ..widgets import BrowserButton,NumericalLineEdit,PathLineEdit

TH1_DEFAULT = 3.00
TH2_DEFAULT = 3.00
MAX_SCAT_DEFAULT = 0.00
MIN_FWHM_DEFAULT = 0.00

class CheckGroup(QtWidgets.QWidget):

    run = QtCore.Signal(str,str,list)

    def __init__(self,base_path,workdir=None,parent=None):

        super().__init__(parent)

        self.base_path = base_path
        self.exec = 'check_group'
        self._setup_ui()
        self._init()
        self._set_connections()

        if workdir is None or not os.path.isdir(workdir):
            self.bt_browse.directory = os.path.getcwd()
        self.bt_browse.directory = workdir

    def _init(self):

        self.le_th1.setText(f'{TH1_DEFAULT:>6.2f}')
        self.le_th2.setText(f'{TH2_DEFAULT:>6.2f}')
        self.le_maxscat.setText(f'{MAX_SCAT_DEFAULT:>6.2f}')
        self.le_minfwhm.setText(f'{MIN_FWHM_DEFAULT:>6.2f}')
        self.le_maxscat.setEnabled(False)
        self.le_minfwhm.setEnabled(False)
        self.cb_centred.setChecked(True)
        self.bt_run.setEnabled(False)

    def _setup_ui(self):

        m = screeninfo.get_monitors()[0]
        scale = round(m.width / 1920)
        dir_icons = os.path.join(self.base_path,'icons')
        self.setWindowIcon(QtGui.QIcon(os.path.join(dir_icons,'checkgroup.svg')))
        self.setWindowTitle('CheckGroup')
        self.setWindowFlags(QtCore.Qt.Window)

        self.bt_browse = BrowserButton.BrowserButton()
        self.bt_cancel = QtWidgets.QPushButton('Cancel')
        self.bt_help = QtWidgets.QPushButton('Help')
        self.bt_run = QtWidgets.QPushButton('Run')
        self.cb_centred = QtWidgets.QCheckBox('Check centred lattices')
        self.cmb_format = QtWidgets.QComboBox()
        self.cmb_format.addItems(['Single Crystal (Shelx)','Single Crystal (Free)',
            'Single Crystal (CIF)','Powder (Integrated Intensities)'])
        self.cmb_system = QtWidgets.QComboBox()
        self.cmb_system.addItems(['Monoclinic','Orthorhombic','Tetragonal',
            'Trigonal / Rombohedral','Hexagonal','Cubic'])
        self.lb_cell = QtWidgets.QLabel('Cell parameters (a,b,c,\u03B1,\u03B2,\u03B3):')
        self.lb_code = QtWidgets.QLabel('File code:')
        self.lb_file = QtWidgets.QLabel('File:')
        self.lb_format = QtWidgets.QLabel('Format:')
        self.lb_lambda = QtWidgets.QLabel('Lambda:')
        self.lb_maxscat = QtWidgets.QLabel('Maximum value of the scattering variable:')
        self.lb_minfwhm = QtWidgets.QLabel('Minimum FWHM distance between two adjacent reflections:')
        self.lb_system = QtWidgets.QLabel('Crystallographic system:')
        self.lb_th1 = QtWidgets.QLabel('Observed reflection threshold as % maximum intensity:')
        self.lb_th2 = QtWidgets.QLabel('Observed reflection threshold as I > n * \u03C3(I):')
        self.lb_title = QtWidgets.QLabel('Title:')
        self.le_cell = QtWidgets.QLineEdit()
        self.le_code = QtWidgets.QLineEdit()
        self.le_file = PathLineEdit.PathLineEdit()
        self.le_lambda = NumericalLineEdit.NumericalLineEdit(scale*100)
        self.le_maxscat = NumericalLineEdit.NumericalLineEdit(scale*100)
        self.le_minfwhm = NumericalLineEdit.NumericalLineEdit(scale*100)
        self.le_th1 = NumericalLineEdit.NumericalLineEdit(scale*100)
        self.le_th2 = NumericalLineEdit.NumericalLineEdit(scale*100)
        self.le_title = QtWidgets.QLineEdit()
        self.gb_info = QtWidgets.QGroupBox('General Information')
        self.gb_pars = QtWidgets.QGroupBox('Parameters')
        self.gb_refs = QtWidgets.QGroupBox('Reflections')

        gl = QtWidgets.QGridLayout()
        gl.addWidget(self.lb_code,0,0)
        gl.addWidget(self.le_code,0,1)
        gl.addWidget(self.lb_title,1,0)
        gl.addWidget(self.le_title,1,1,1,5)
        gl.addWidget(self.lb_lambda,2,0)
        gl.addWidget(self.le_lambda,2,1)
        gl.addWidget(self.lb_system,2,2)
        gl.addWidget(self.cmb_system,2,3)
        gl.addWidget(self.cb_centred,2,4)
        gl.setColumnStretch(5,10)
        self.gb_info.setLayout(gl)

        gl = QtWidgets.QGridLayout()
        gl.addWidget(self.lb_file,0,0)
        gl.addWidget(self.le_file,0,1,1,2)
        gl.addWidget(self.bt_browse,0,3)
        gl.addWidget(self.lb_format,1,0)
        gl.addWidget(self.cmb_format,1,1)
        gl.setColumnStretch(2,10)
        self.gb_refs.setLayout(gl)

        gl = QtWidgets.QGridLayout()
        gl.addWidget(self.lb_th1,0,0)
        gl.addWidget(self.le_th1,0,1)
        gl.addWidget(self.lb_th2,0,2)
        gl.addWidget(self.le_th2,0,3)
        gl.addWidget(self.lb_maxscat,1,0)
        gl.addWidget(self.le_maxscat,1,1)
        gl.addWidget(self.lb_minfwhm,1,2)
        gl.addWidget(self.le_minfwhm,1,3)
        gl.setColumnStretch(4,10)
        self.gb_pars.setLayout(gl)

        hl = QtWidgets.QHBoxLayout()
        hl.addWidget(self.bt_help)
        hl.addStretch()
        hl.addWidget(self.bt_run)
        hl.addWidget(self.bt_cancel)

        vl = QtWidgets.QVBoxLayout()
        vl.addWidget(self.gb_info)
        vl.addWidget(self.gb_refs)
        vl.addWidget(self.gb_pars)
        vl.addLayout(hl)
        vl.addStretch()

        self.setLayout(vl)

    def _set_connections(self):

        self.bt_browse.fileSelected.connect(self.le_file.set_full_path)
        self.bt_cancel.clicked.connect(self.close)
        self.cmb_format.currentIndexChanged.connect(self.update_ui)

    def update_ui(self,index):

        if index == 3:
            enabled = True
        else:
            enabled = False
        self.le_maxscat.setEnabled(enabled)
        self.le_minfwhm.setEnabled(enabled)